
import numpy as np
import matplotlib.pyplot as plt

# Constants
SOLAR_MASS_KG = 1.989e30
SgrA_OBSERVED_MASS_SOLAR = 4.3e6
SgrA_OBSERVED_MASS_KG = SgrA_OBSERVED_MASS_SOLAR * SOLAR_MASS_KG
CMAX_BITS_SgrA = 2.80e90

# Time-normalized RC curve
time_norm = np.linspace(0, 1, 1000)
RC_curve = CMAX_BITS_SgrA * (1 - np.exp(-5 * time_norm))

# Convert rendered capacity to mass
rendered_bits = RC_curve
mass_estimate_kg = (rendered_bits / CMAX_BITS_SgrA) * SgrA_OBSERVED_MASS_KG
mass_estimate_solar = mass_estimate_kg / SOLAR_MASS_KG

# Final rendered mass and percent error
final_mass_model = mass_estimate_solar[-1]
percent_error = abs((final_mass_model - SgrA_OBSERVED_MASS_SOLAR) / SgrA_OBSERVED_MASS_SOLAR) * 100

# Redshift mapping (approximate)
redshift_ticks = np.linspace(0, 1, 6)
redshift_labels = [f"z ≈ {int((1 - t) * 15)}" for t in redshift_ticks]

# Plot
fig, ax1 = plt.subplots(figsize=(10, 6))
ax1.plot(time_norm, mass_estimate_solar, label="Mass Estimate (Solar Masses)", color='green')
ax1.fill_between(time_norm, mass_estimate_solar, color='green', alpha=0.2)
ax1.axhline(y=SgrA_OBSERVED_MASS_SOLAR, color='red', linestyle='--', label="Observed Mass (Sagittarius A*)")

ax1.set_title("Sagittarius A* Mass Evolution from RC Load Curve")
ax1.set_xlabel("Normalized Time")
ax1.set_ylabel("Mass (Solar Masses)")
ax1.legend()
ax1.grid(True)

# Annotation
annotation = (
    f"Observed Mass: {SgrA_OBSERVED_MASS_SOLAR:.2e} M☉\n"
    f"Model Mass: {final_mass_model:.2e} M☉\n"
    f"Error: {percent_error:.2f}%"
)
ax1.text(0.6, 0.2 * SgrA_OBSERVED_MASS_SOLAR, annotation, fontsize=10,
         bbox=dict(facecolor='white', edgecolor='gray', boxstyle='round'))

# Redshift axis on top
ax2 = ax1.twiny()
ax2.set_xlim(ax1.get_xlim())
ax2.set_xticks(redshift_ticks)
ax2.set_xticklabels(redshift_labels)
ax2.set_xlabel("Approximate Redshift")

plt.tight_layout()
plt.show()
